%%
% To find the standing wave peak positions for christmas tree from COMSOL 
% band structure for a given effective reflection and transmission 
% co-efficient at the corners
% 
% Inputs: ct.tri_sideL: effective circumference/3, Gamma, ct.horz_bot:
%         horizontal bottom length, ct.slant_bot: slanted bottom length
%         (All lengths are in number of unit cells)
%         Comsol band structures: experiment_compare_data_horizontal.mat, 
%         experiment_compare_data_slant.mat
%         Transmission and reflected co-efficients: (scat.t1, scat.r1) for horizontal
%         to slanted, (scat.t2, scat.r2) for slanted to slanted
% Outputs: Fig 1: function a in eigenvalue characteristic equation 
%          x^2+ax+1=0 (a=-2 is the peak position)
%          Fig 2: Lines at the peak position
%          Fig 3 and Fig 4: expected spectrum at horizontal and slanted
%          side respectively
%%

format long;

clear all;
close all;

% Load band structure data values
load('../Data/experiment_compare_data_horizontal.mat');
load('../Data/experiment_compare_data_slant.mat');

% Get the range of energy values
energy_min = max(min(data_horz.energy), min(data_slant.energy));
energy_max = min(max(data_horz.energy), max(data_slant.energy));

% Range of values of energy to check out for peak position
num_energy = 5e6;
energy = linspace(energy_min, energy_max, num_energy);

% Find the k value for each energy via interpolation
kinterp.horz = interp1(data_horz.energy, data_horz.k, energy, 'spline');
kinterp.slant = interp1(data_slant.energy, data_slant.k, energy, 'spline');

% Get g(beta) for each energy
ct.amech = 1.570496020158940e-05;

% Christmas tree parameters (length in number of snowflake unit cells)
ct.realL = 32;
% ct.tri_side = 25.628000572; % Slanted_v1 circumference length/3
ct.tri_side = 25.575950949; % Horizontal_v1 circumference length/3
ct.horz_bot = 14/32*ct.tri_side; % Horizontal bottom length
ct.slant_bot = 6/32*ct.tri_side; % Slanted bottom length

beta1 = kinterp.horz*ct.horz_bot*ct.amech;
beta2 = kinterp.slant*ct.slant_bot*ct.amech;
beta3 = kinterp.horz*(ct.tri_side - ct.horz_bot)/2*ct.amech;
beta4 = kinterp.slant*(ct.tri_side - ct.slant_bot)*ct.amech;

% Reflection and transmission co-efficients
scat.t1 = 0.6; scat.r1 = 1i*sqrt(1 - abs(scat.t1)^2);
scat.t2 = 0.6; scat.r2 = 1i*sqrt(1 - abs(scat.t2)^2);

% Peak index
peak_index = 15;

% Quadratic equation of the form: x^2+ax+1=0
a = 1/(scat.t1^6.*scat.t2.*conj(scat.t1)^6.*conj(scat.t2)).*exp(-i.*(beta1+2.*(beta2+beta3+beta4))).*...
      (-exp(2i.*(beta1+2.*(beta2+beta3+beta4))).*scat.t1^6.*scat.t2+exp(4i.*(beta2+beta4)).*...
     scat.t1^6.*scat.t2.*conj(scat.r1)^6+2.*exp(2i.*(beta1+beta2+2.*(beta3+beta4))).*...
     scat.r1^2.*scat.t1^4.*scat.t2.*conj(scat.t1)^2+exp(4i.*(beta3+beta4)).*scat.r1^2.*scat.t1^2.*scat.t2.*...
     conj(scat.t1)^4-exp(2i.*(beta1+2.*(beta3+beta4))).*scat.r1^4.*scat.t1^2.*scat.t2.*conj(scat.t1)^4-...
      exp(2i.*(beta3+beta4)).*scat.r1.*scat.t1.*scat.t2.*conj(scat.r2).*conj(scat.t1).*...
      (exp(2i.*(beta1+2.*beta2+beta3)).*scat.t1^4-exp(2i.*(beta1+beta2)).*...
      (-1+2.*exp(2i.*beta3).*scat.r1^2).*scat.t1^2.*conj(scat.t1)^2+(-1+exp(2i.*beta1).*scat.r1^2).*...
      (-1+exp(2i.*beta3).*scat.r1^2).*conj(scat.t1)^4)+exp(2i.*(beta1+2.*beta2+2.*beta3+beta4)).*...
     scat.r1.*scat.r2.*scat.t1^5.*conj(scat.t1).*conj(scat.t2)+exp(2i.*(beta1+2.*(beta2+beta3))).*scat.r1^2.*...
     scat.t1^4.*conj(scat.t1)^2.*conj(scat.t2)+exp(2i.*(beta1+beta2+beta3+beta4)).*scat.r1.*scat.r2.*...
     scat.t1^3.*conj(scat.t1)^3.*conj(scat.t2)-2.*exp(2i.*(beta1+beta2+2.*beta3+beta4)).*scat.r1^3.*...
      scat.r2.*scat.t1^3.*conj(scat.t1)^3.*conj(scat.t2)+2.*exp(2i.*(beta1+beta2+beta3)).*scat.r1^2.*scat.t1^2.*...
     conj(scat.t1)^4.*conj(scat.t2)-2.*exp(2i.*(beta1+beta2+2.*beta3)).*scat.r1^4.*scat.t1^2.*...
     conj(scat.t1)^4.*conj(scat.t2)+exp(2i.*(beta3+beta4)).*scat.r1.*scat.r2.*scat.t1.*conj(scat.t1)^5.*conj(scat.t2)-...
      exp(2i.*(beta1+beta3+beta4)).*scat.r1^3.*scat.r2.*scat.t1.*conj(scat.t1)^5.*conj(scat.t2)-...
      exp(2i.*(2.*beta3+beta4)).*scat.r1^3.*scat.r2.*scat.t1.*conj(scat.t1)^5.*conj(scat.t2)+...
      exp(2i.*(beta1+2.*beta3+beta4)).*scat.r1^5.*scat.r2.*scat.t1.*conj(scat.t1)^5.*conj(scat.t2)-...
     conj(scat.t1)^6.*conj(scat.t2)+exp(2i.*beta1).*scat.r1^2.*conj(scat.t1)^6.*conj(scat.t2)+...
      2.*exp(2i.*beta3).*scat.r1^2.*conj(scat.t1)^6.*conj(scat.t2)-exp(4i.*beta3).*scat.r1^4.*conj(scat.t1)^6.*...
     conj(scat.t2)-2.*exp(2i.*(beta1+beta3)).*scat.r1^4.*conj(scat.t1)^6.*conj(scat.t2)+...
      exp(2i.*(beta1+2.*beta3)).*scat.r1^6.*conj(scat.t1)^6.*conj(scat.t2)+...
      exp(2i.*(2.*beta2+beta4)).*scat.t1^5.*conj(scat.r1)^5.*conj(scat.t1).*(scat.t2.*conj(scat.r2)-...
      scat.r2.*conj(scat.t2))-exp(2i.*beta2).*scat.t1^4.*conj(scat.r1)^4.*...
      (exp(2i.*(beta2+2.*beta4)).*(exp(2i.*beta1)+2.*exp(2i.*beta3)).*scat.t1^2.*scat.t2+...
      exp(2i.*(beta2+beta3+beta4)).*scat.r1.*scat.t1.*scat.t2.*conj(scat.r2).*conj(scat.t1)-...
      exp(2i.*(beta2+beta3+beta4)).*scat.r1.*scat.r2.*scat.t1.*conj(scat.t1).*conj(scat.t2)+conj(scat.t1)^2.*...
      (2.*exp(4i.*beta4).*scat.t2+exp(2i.*beta2).*conj(scat.t2)))+exp(2i.*beta2).*...
     scat.t1^3.*conj(scat.r1)^3.*conj(scat.t1).*(-exp(2i.*beta4).*scat.t2.*conj(scat.r2).*(exp(2i.*beta2).*...
      (exp(2i.*beta1)+exp(2i.*beta3)).*scat.t1^2+(2-exp(2i.*beta3).*scat.r1^2).*conj(scat.t1)^2)+...
      exp(2i.*(beta2+beta4)).*(exp(2i.*beta1)+exp(2i.*beta3)).*scat.r2.*scat.t1^2.*conj(scat.t2)-...
      exp(2i.*beta4).*(-2+exp(2i.*beta3).*scat.r1^2).*scat.r2.*conj(scat.t1)^2.*conj(scat.t2)+...
      2.*scat.r1.*scat.t1.*conj(scat.t1).*(exp(4i.*beta4).*(exp(2i.*beta1)+exp(2i.*beta3)).*scat.t2+...
      exp(2i.*(beta2+beta3)).*conj(scat.t2)))-scat.t1.*conj(scat.r1).*conj(scat.t1).*...
      (exp(2i.*beta4).*scat.t2.*conj(scat.r2).*(-exp(2i.*(beta1+2.*beta2+beta3)).*scat.t1^4+...
      exp(2i.*(beta2+beta3)).*(-1+2.*exp(2i.*beta1).*scat.r1^2+2.*exp(2i.*beta3).*scat.r1^2).*...
      scat.t1^2.*conj(scat.t1)^2-(-1+exp(2i.*beta1).*scat.r1^2).*(-1+exp(2i.*beta3).*scat.r1^2).*conj(scat.t1)^4)+...
      exp(2i.*(beta1+2.*beta2+beta3+beta4)).*scat.r2.*scat.t1^4.*conj(scat.t2)-exp(2i.*(beta2+beta3+beta4)).*...
      (-1+2.*exp(2i.*beta1).*scat.r1^2+2.*exp(2i.*beta3).*scat.r1^2).*scat.r2.*scat.t1^2.*conj(scat.t1)^2.*conj(scat.t2)+...
      exp(2i.*beta4).*(-1+exp(2i.*beta1).*scat.r1^2).*(-1+exp(2i.*beta3).*scat.r1^2).*scat.r2.*conj(scat.t1)^4.*...
      conj(scat.t2)+2.*exp(2i.*(beta2+beta3)).*scat.r1.*scat.t1^3.*conj(scat.t1).*(exp(4i.*beta4).*...
      (exp(2i.*beta1)+exp(2i.*beta3)).*scat.t2+exp(2i.*(beta1+beta2)).*conj(scat.t2))-...
      2.*scat.r1.*scat.t1.*conj(scat.t1)^3.*(exp(2i.*(beta3+2.*beta4)).*(-1+exp(2i.*beta1).*scat.r1^2).*scat.t2+...
      exp(2i.*beta2).*(exp(2i.*beta1)+exp(2i.*beta3)).*(-1+exp(2i.*beta3).*scat.r1^2).*...
      conj(scat.t2)))+scat.t1^2.*conj(scat.r1)^2.*(exp(2i.*(2.*beta2+beta3+2.*beta4)).*...
      (2.*exp(2i.*beta1)+exp(2i.*beta3)).*scat.t1^4.*scat.t2+exp(2i.*(beta2+beta4)).*scat.r1.*scat.t1.*...
      scat.t2.*conj(scat.r2).*conj(scat.t1).*(exp(2i.*(beta2+beta3)).*(exp(2i.*beta1)+...
      exp(2i.*beta3)).*scat.t1^2+(2.*exp(2i.*beta1)+2.*exp(2i.*beta3)-exp(2i.*(beta1+beta3)).*...
      scat.r1^2).*conj(scat.t1)^2)-exp(2i.*(2.*beta2+beta3+beta4)).*(exp(2i.*beta1)+...
      exp(2i.*beta3)).*scat.r1.*scat.r2.*scat.t1^3.*conj(scat.t1).*conj(scat.t2)+exp(2i.*(beta2+beta4)).*...
      scat.r1.*(-2.*exp(2i.*beta1)-2.*exp(2i.*beta3)+exp(2i.*(beta1+beta3)).*scat.r1^2).*...
      scat.r2.*scat.t1.*conj(scat.t1)^3.*conj(scat.t2)+conj(scat.t1)^4.*(-exp(4i.*beta4).*(-1+...
      exp(2i.*beta1).*scat.r1^2).*scat.t2-2.*exp(2i.*beta2).*(-1+exp(2i.*beta3).*scat.r1^2).*...
      conj(scat.t2))-exp(2i.*beta2).*scat.t1^2.*conj(scat.t1)^2.*(2.*exp(2i.*(beta3+2.*beta4)).*...
      (-1+exp(2i.*beta1).*scat.r1^2).*scat.t2-exp(2i.*beta2).*(exp(2i.*beta1)-...
      exp(4i.*beta3).*scat.r1^2).*conj(scat.t2))));
  
% Peak positions
[peak_pos, kpeak, vel] = find_roots_ene(energy, real(a)+2, kinterp);

%% Eigenvector calculation

% Measurement node distance (calculated counter-clockwise from right corner)
num_points = 5000;
% dis = linspace(1.28, 2.99, num_points)*ct.tri_side*ct.amech;
dis = (linspace(1,3*ct.realL,num_points)...
    /ct.realL).*ct.L*ct.amech;
amp = zeros(size(peak_pos, 2), num_points);

for i=1:num_points
    % Amplitude at a distance dis
    amp(:,i) = get_amp_eigenvector(kpeak, scat, dis(i), ct, vel);
end

%% Plotting

figure(1)
hold on;
set(gca,'FontSize',18)
pbaspect([8.147 1 1])

basevalue = 0;
% maxvalue = max(max(amp(peak_index:peak_index+1, :)));

% xlim([42 70])
% ylim([basevalue maxvalue])
xlabel('Distance from left bottom corner');
ylabel('Amplitude');
box on;

peak_index = 15;
plot(32*dis/(ct.tri_side*ct.amech), amp(peak_index, :), ...
    'LineWidth', 1, 'Color', 'blue')
% plot(32*dis/(ct.tri_side*ct.amech), amp(peak_index+1, :), ...
%     'LineWidth', 1, 'Color', 'red')